// models/Zone.ts
import mongoose, { Schema, Document, Types } from "mongoose";

interface ITransportFee {
  currency: string;
  amount: number;
}

interface IZone extends Document {
  name: string;
  countries: string[];
  transportFee: ITransportFee;
  createdAt: Date;
  updatedAt: Date;
}

const ZoneSchema = new Schema<IZone>({
  name: { 
    type: String, 
    required: true, 
    unique: true,
    trim: true,
    maxlength: 100
  },
  countries: { 
    type: [String], 
    required: true,
    validate: {
      validator: (countries: string[]) => countries.length > 0,
      message: "At least one country is required"
    }
  }, 
  transportFee: {
    currency: { 
      type: String, 
      required: true,
      uppercase: true,
      enum: ["USD", "EUR", "MAD"],
      default: "EUR"
    },
    amount: { 
      type: Number, 
      required: true,
      min: [0, "Amount cannot be negative"]
    }
  },
  createdAt: { type: Date, default: Date.now },
  updatedAt: { type: Date, default: Date.now }
});

ZoneSchema.pre("save", function (next) {
  this.updatedAt = new Date();
  next();
});

export const Zone = mongoose.model<IZone>("Zone", ZoneSchema);