import express from "express";
import { User } from "../models/User";

// ✅ Middleware: Check if user is authenticated
export const isAuthenticated = async (
  req: express.Request,
  res: express.Response,
  next: express.NextFunction
) => {
  try {
    const sessionToken = req.cookies["sessionToken"];

    if (!sessionToken) {
      return res.status(401).json({ message: "Unauthorized: No session token provided" });
    }

    const existingUser = await User.findOne({ "authentication.sessionToken": sessionToken });

    if (!existingUser) {
      return res.status(401).json({ message: "Unauthorized: Invalid session token" });
    }

    if (!existingUser.isActive) {
      return res.status(403).json({ message: "Forbidden: User is not active" });
    }

    // Attach user to req.user
    req.user = {
      _id: existingUser._id,
      role: existingUser.role, // Ensure the role is added here
      isActive: existingUser.isActive,
    };

    next();
  } catch (error) {
    console.error("Error in isAuthenticated middleware:", error);
    return res.status(400).json({ message: "Bad request (Check if user is authenticated)" });
  }
};


// ✅ Middleware: Check if user is SuperAdmin
export const isSuperAdmin = async (
  req: express.Request,
  res: express.Response,
  next: express.NextFunction
) => {
  try {
    // Check if user is SuperAdmin
    if (!req.user || req.user.role !== "SuperAdmin") {
      return res.status(403).json({ message: "Forbidden: Requires SuperAdmin access" });
    }
    next();
  } catch (error) {
    console.error("Error in isSuperAdmin middleware:", error);
    return res.status(400).json({ message: "Bad request (Check if user is SuperAdmin)" });
  }
};
