// controllers/zoneController.ts
import { Request, Response } from "express";
import { Zone } from "../models/Zone";
import { Log } from "../models/Log";
import { Notification } from "../models/Notification";
import { User } from "../models/User";
import mongoose from "mongoose";

const formatResponse = (success: boolean, data?: any, message?: string) => ({
  success,
  ...(data && { data }),
  ...(message && { message }),
});

// Create Zone
export const createZone = async (req: Request, res: Response) => {
  const session = await mongoose.startSession();
  session.startTransaction();

  try {
    const { name, countries, transportFee } = req.body;
    const userId = req.headers["user-id"] as string;

    // Add after userId validation
    const userExists = await User.exists({ _id: userId }).session(session);
    if (!userExists) {
      return res.status(400).json(formatResponse(false, null, "User not found"));
    }

    // Validation
    if (!userId || !mongoose.Types.ObjectId.isValid(userId)) {
      return res.status(400).json(formatResponse(false, null, "Invalid User ID"));
    }

    const requiredFields = ['name', 'countries', 'transportFee'];
    const missingFields = requiredFields.filter(field => !req.body[field]);
    if (missingFields.length > 0) {
      return res.status(400).json(formatResponse(false, null, `Missing required fields: ${missingFields.join(', ')}`));
    }

    // Create zone
    const newZone = new Zone({
      name,
      countries,
      transportFee
    });

    await newZone.save();

    await Log.create({
      action: "CREATE_ZONE_SHIPPING",
      performedBy: new mongoose.Types.ObjectId(userId),
      details: { name, countries, transportFee },
    });

    // Notify admins
    const admins = await User.find({ role: "SuperAdmin" });
    await Promise.all(
      admins.map(async (admin) => {
        await Notification.create({
          userId: admin._id,
          title: "New Zone shipping Created",
          message: `Zone '${name}' created with ${countries.length || 0} countries`,
          type: "info",
        });
      })
    );
    return res.status(201).json(formatResponse(true, newZone));

  } catch (error) {
    await session.abortTransaction();
    console.error("Zone creation error:", error);
    return res.status(500).json(formatResponse(false, null, "Internal server error"));
  } finally {
    session.endSession();
  }
};

// Get All Zones
export const getAllZones = async (req: Request, res: Response) => {
  try {
    const { search } = req.query;
    const userRole = req.headers["user-role"] as string;

    if (userRole !== "SuperAdmin" && userRole !== "Photographer") {
      return res.status(403).json(formatResponse(false, null, "Unauthorized access"));
    }

    const filter: any = {};
    if (search) {
      filter.$or = [
        { name: { $regex: search, $options: 'i' } },
        { countries: { $elemMatch: { $regex: search, $options: 'i' } } }
      ];
    }

    const zones = await Zone.find(filter).sort({ createdAt: -1 });

    return res.json(formatResponse(true, zones));

  } catch (error) {
    console.error("Fetch error:", error);
    return res.status(500).json(formatResponse(false, null, "Internal server error"));
  }
};

// Get Zone by ID
export const getZoneById = async (req: Request, res: Response) => {
  try {
    const { id } = req.params;
    const userRole = req.headers["user-role"] as string;

    if (userRole !== "SuperAdmin" && userRole !== "Photographer") {
      return res.status(403).json(formatResponse(false, null, "Unauthorized access"));
    }

    if (!mongoose.Types.ObjectId.isValid(id)) {
      return res.status(400).json(formatResponse(false, null, "Invalid Zone ID"));
    }

    const zone = await Zone.findById(id);
    if (!zone) {
      return res.status(404).json(formatResponse(false, null, "Zone not found"));
    }

    return res.json(formatResponse(true, zone));

  } catch (error) {
    console.error("Fetch error:", error);
    return res.status(500).json(formatResponse(false, null, "Internal server error"));
  }
};

// Update Zone
export const updateZone = async (req: Request, res: Response) => {
  const session = await mongoose.startSession();
  session.startTransaction();

  try {
    const { id } = req.params;
    const userId = req.headers["user-id"] as string;
    const updates = req.body;

    if (!mongoose.Types.ObjectId.isValid(id) || !mongoose.Types.ObjectId.isValid(userId)) {
      return res.status(400).json(formatResponse(false, null, "Invalid ID format"));
    }

    const zone = await Zone.findByIdAndUpdate(
      id,
      { $set: updates, $currentDate: { updatedAt: true } },
      { new: true, session }
    );

    if (!zone) {
      return res.status(404).json(formatResponse(false, null, "Zone not found"));
    }

    await Log.create([{
      action: "ZONE_UPDATE",
      performedBy: new mongoose.Types.ObjectId(userId),
      targetId: zone._id,
      details: updates
    }], { session });

    await session.commitTransaction();
    return res.json(formatResponse(true, zone));

  } catch (error) {
    await session.abortTransaction();
    console.error("Update error:", error);
    return res.status(500).json(formatResponse(false, null, "Internal server error"));
  } finally {
    session.endSession();
  }
};

// Delete Zone
export const deleteZone = async (req: Request, res: Response) => {
  const session = await mongoose.startSession();
  session.startTransaction();

  try {
    const { id } = req.params;
    const userId = req.headers["user-id"] as string;

    if (!mongoose.Types.ObjectId.isValid(id) || !mongoose.Types.ObjectId.isValid(userId)) {
      return res.status(400).json(formatResponse(false, null, "Invalid ID format"));
    }

    const deletedZone = await Zone.findByIdAndDelete(id, { session });
    if (!deletedZone) {
      return res.status(404).json(formatResponse(false, null, "Zone not found"));
    }

    await Log.create([{
      action: "ZONE_DELETE",
      performedBy: new mongoose.Types.ObjectId(userId),
      targetId: deletedZone._id,
      details: { name: deletedZone.name }
    }], { session });

    await session.commitTransaction();
    return res.json(formatResponse(true, null, "Zone deleted successfully"));

  } catch (error) {
    await session.abortTransaction();
    console.error("Deletion error:", error);
    return res.status(500).json(formatResponse(false, null, "Internal server error"));
  } finally {
    session.endSession();
  }
};