import { Request, Response } from 'express';
import { User } from '../models/User';
import bcrypt from 'bcrypt';
import { Log } from '../models/Log';
import { Notification } from '../models/Notification';

export const getAllUsers = async (req: Request, res: Response) => {
  try {
    // Extract userId and userRole from request headers
    const userId = req.headers["user-id"] as string; // Access userId from headers
    const userRole = req.headers["user-role"] as string; // Access userRole from headers

    if (!userId) {
      return res.status(401).json({ message: "Unauthorized: No user ID provided" });
    }

    // If the user is SuperAdmin, show all notifications
    if (userRole === "SuperAdmin") {
      const users = await User.find().sort({ createdAt: -1 });
      return res.status(200).json(users);
    }
    const users = await User.find({ userId }).sort({ createdAt: -1 });
    return res.status(200).json(users);
  } catch (err) {
    res.status(500).json({ error: (err as Error).message });
  }
};

// Add this to your user controller file
export const changePassword = async (req: Request, res: Response) => {
  try {
    const { currentPassword, newPassword } = req.body;

    const storedUserData = sessionStorage.getItem("userData");
    if (!storedUserData) return;

    const userData = JSON.parse(storedUserData);
    const userId = userData._id;

    if (!userId) {
      return res.status(401).json({ message: "Unauthorized" });
    }

    if (!currentPassword || !newPassword) {
      return res.status(400).json({ message: "All fields are required" });
    }

    if (newPassword.length < 6) {
      return res.status(400).json({ message: "Password must be at least 6 characters" });
    }

    const user = await User.findById(userId);
    if (!user || !user.authentication?.password) {
      return res.status(404).json({ message: "User not found" });
    }

    const isMatch = await bcrypt.compare(currentPassword, user.authentication.password);
    if (!isMatch) {
      return res.status(401).json({ message: "Current password is incorrect" });
    }

    const salt = await bcrypt.genSalt(10);
    const hashedPassword = await bcrypt.hash(newPassword, salt);

    user.authentication.password = hashedPassword;
    user.authentication.salt = salt;
    await user.save();

    await Log.create({
      action: "PASSWORD_CHANGE",
      performedBy: user._id,
      details: { changedAt: new Date() },
    });

    await Notification.create({
      userId: user._id,
      title: "Password Changed",
      message: "Your password was successfully updated.",
      type: "security",
    });

    return res.status(200).json({ message: "Password updated successfully" });
  } catch (error) {
    console.error("Error changing password:", error);
    return res.status(500).json({ message: "Internal server error" });
  }
};

// Get user by ID
export const getUsersById = async (req: Request, res: Response) => {
  try {
    const { id } = req.params;
    const user = await User.findById(id);
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }
    res.json(user);
  } catch (error) {
    res.status(500).json({ error: (error as Error).message });
  }
};

// Get user by session token
export const getUsersBySessionToken = async (sessionToken: string) => {
  try {
    return await User.findOne({ 'authentication.sessionToken': sessionToken });
  } catch (error) {
    throw new Error('User not found');
  }
};

// Get users by role (only active users)
export const getUsersByRole = async (req: Request, res: Response) => {
  try {
    const { role } = req.params;
    const validRoles = ["SuperAdmin", "Photographer", "Infographiste", "Imprimante"];

    if (!validRoles.includes(role)) {
      return res.status(400).json({ error: "Invalid role" });
    }

    // Fetch only active users with the specified role
    const users = await User.find({ role }).populate("SalesZone");

    if (users.length === 0) {
      return res.status(404).json({ message: `No active users found with role "${role}"` });
    }

    res.json(users);
  } catch (err) {
    res.status(500).json({ error: (err as Error).message });
  }
};

// Delete user by ID
export const deleteUser = async (req: Request, res: Response) => {
  try {
    const { id } = req.params;
    const deletedUser = await User.findByIdAndDelete(id);
    if (!deletedUser) {
      return res.status(404).json({ error: 'User not found' });
    }
    res.json({ message: 'User deleted successfully' });
  } catch (error) {
    res.status(500).json({ error: (error as Error).message });
  }
};

// Update user by ID
export const updateUser = async (req: Request, res: Response) => {
  try {
    const { id } = req.params;
    const { username } = req.body;

    if (!username) {
      return res.status(400).json({ error: 'Username is required' });
    }

    const updatedUser = await User.findByIdAndUpdate(id, { username }, { new: true });
    if (!updatedUser) {
      return res.status(404).json({ error: 'User not found' });
    }
    res.json(updatedUser);
  } catch (error) {
    res.status(500).json({ error: (error as Error).message });
  }
};
