import { Request, Response } from "express";
// src/controllers/statusController.ts
import { Status } from '../models/Status'; // Correct import path
import { Log } from "../models/Log";
import { Notification } from "../models/Notification";
import { User } from "../models/User";
import { File } from "../models/File";
import mongoose from 'mongoose';


// controllers/status.ts
export const createStatus = async (req: Request, res: Response) => {
  try {
    const { status, type, paymentMethod, updatedBy } = req.body;

    const newStatus = new Status({
      status,
      type,
      paymentMethod,
      updatedBy,
    });

    await newStatus.save();

    return res.status(201).json(newStatus);
  } catch (error) {
    console.error("Error creating status:", error);
    return res.status(500).json({ message: "Failed to create status" });
  }
};

// Get all statuses
export const getAllStatuses = async (_req: Request, res: Response) => {
  try {
    const statuses = await Status.find().populate(["file", "updatedBy"]);
    return res.status(200).json(statuses);
  } catch (error) {
    console.error("Error fetching statuses:", error);
    return res.status(500).json({ message: "Failed to fetch statuses" });
  }
};

// Get status by ID
export const getStatusById = async (req: Request, res: Response) => {
  try {
    const status = await Status.findById(req.params.id).populate(["file", "updatedBy"]);
    if (!status) {
      return res.status(404).json({ message: "Status not found" });
    }
    return res.status(200).json(status);
  } catch (error) {
    console.error("Error fetching status:", error);
    return res.status(500).json({ message: "Failed to fetch status" });
  }
};

export const updateStatus = async (req: Request, res: Response) => {
  const { id } = req.params;
  const { status, type, paymentMethod, updatedBy } = req.body;

  // Validate the ID
  if (!mongoose.Types.ObjectId.isValid(id)) {
    return res.status(400).json({ message: "Invalid status ID" });
  }

  try {
    const existingStatus = await Status.findById(id);
    if (!existingStatus) {
      return res.status(404).json({ message: "Status not found" });
    }

    const updatedStatus = await Status.findByIdAndUpdate(
      id,
      { status, type, paymentMethod, updatedBy, updatedAt: new Date() },
      { new: true }
    ).populate(["file", "updatedBy"]);

    if (!updatedStatus) {
      return res.status(404).json({ message: "Status not found after update" });
    }

    // Read User-Id from headers
    const userId = req.headers["user-id"] as string;

    // Validate User-Id
    if (!userId || !mongoose.Types.ObjectId.isValid(userId)) {
      return res.status(400).json({ message: "Invalid or missing User-Id in headers" });
    }

    // Log status update
    await Log.create({
      action: "UPDATE_STATUS",
      performedBy: userId, // Use the User-Id from headers
      details: {
        fileId: existingStatus.file,
        newStatus: status,
        domicileType: type,
      },
    });

    // Notify assigned user and admins
    const file = await File.findById(existingStatus.file);
    if (file?.assignedTo) {
      await Notification.create({
        userId: file.assignedTo,
        title: "Project Status Updated",
        message: `The project ${file.clientName} (${type}) is now marked as ${status}.`,
        type: "info",
      });
    }

    const admins = await User.find({ role: "SuperAdmin" });
    for (const admin of admins) {
      await Notification.create({
        userId: admin._id,
        title: "Project Status Changed",
        message: `A project status (${type}) has been updated to ${status}.`,
        type: "info",
      });
    }

    return res.status(200).json({ message: "Status updated successfully", status: updatedStatus });
  } catch (error) {
    console.error("Error updating status:", error);

    // Safely handle the error object
    let errorMessage = "Failed to update status";
    if (error instanceof Error) {
      errorMessage = error.message;
    }

    return res.status(500).json({ message: errorMessage });
  }
};

// Delete status (if needed)
export const deleteStatus = async (req: Request, res: Response) => {
  const { id } = req.params;
  try {
    const deletedStatus = await Status.findByIdAndDelete(id);
    if (!deletedStatus) {
      return res.status(404).json({ message: "Status not found" });
    }

    // Log deletion
    await Log.create({
      action: "DELETE_STATUS",
      performedBy: req.user?._id,
      details: {
        fileId: deletedStatus.file,
        domicileType: deletedStatus.type // Added type to deletion logs
      },
    });

    return res.status(200).json({ message: "Status deleted successfully" });
  } catch (error) {
    console.error("Error deleting status:", error);
    return res.status(500).json({ message: "Failed to delete status" });
  }
};