import { Request, Response } from "express";
import { Statistics } from "../models/Statistics";
import { Log } from "../models/Log";
import { Notification } from "../models/Notification";
import { User } from "../models/User";
import mongoose from "mongoose";
import { SalesZone } from "src/models/SalesZone";
import { Status } from "src/models/Status";

// ========== Controllers ========== //
// In your file controller
export const getFileStatistics = async (req: Request, res: Response) => {
  try {
    const userId = req.headers["user-id"] as string;
    const userRole = req.headers["user-role"] as string;

    if (!userId || !mongoose.Types.ObjectId.isValid(userId)) {
      return res.status(400).json({ message: "Invalid User-Id" });
    }

    // Build base query
    let matchQuery: any = {};

    // Photographer filter
    if (userRole === "Photographer") {
      const salesZones = await SalesZone.find({ photographers: userId }).select("_id");
      const salesZoneIds = salesZones.map(zone => zone._id);
      if (salesZoneIds.length === 0) return res.status(200).json(getEmptyStats());
      matchQuery.salesZone = { $in: salesZoneIds };
    }

    // Imprimante filter
    if (userRole === "Imprimante") {
      const paidStatus = await Status.findOne({ status: "Paid" });
      if (!paidStatus) return res.status(200).json(getEmptyStats());
      matchQuery.status = paidStatus._id;
    }

    // Aggregation pipeline
    const pipeline = [
      { $match: matchQuery },
      {
        $lookup: {
          from: "statuses",
          localField: "status",
          foreignField: "_id",
          as: "statusInfo"
        }
      },
      { $unwind: "$statusInfo" },
      {
        $group: {
          _id: null,
          total: { $sum: 1 },
          paid: {
            $sum: { $cond: [{ $eq: ["$statusInfo.status", "Paid"] }, 1, 0] }
          },
          pending: {
            $sum: { $cond: [{ $eq: ["$statusInfo.status", "Pending"] }, 1, 0] }
          },
          inProgress: {
            $sum: { $cond: [{ $eq: ["$statusInfo.status", "In Progress"] }, 1, 0] }
          },
          completed: {
            $sum: { $cond: [{ $eq: ["$statusInfo.status", "Completed"] }, 1, 0] }
          }
        }
      },
      {
        $project: {
          _id: 0,
          total: 1,
          paid: 1,
          pending: 1,
          inProgress: 1,
          completed: 1
        }
      }
    ];

    const stats = await File.aggregate(pipeline);
    
    // Log successful request
    await Log.create({
      action: "FETCH_STATS",
      performedBy: userId,
      details: { userRole, stats: stats[0] || {} }
    });

    return res.status(200).json(stats[0] || getEmptyStats());

  } catch (error) {
    console.error("Statistics Error:", error);
    return res.status(500).json({ 
      message: "Internal server error",
      error: error instanceof Error ? error.message : "Unknown error"
    });
  }
};

// Helper function
const getEmptyStats = () => ({
  total: 0,
  paid: 0,
  pending: 0,
  inProgress: 0,
  completed: 0
});

export const updateStatistics = async (req: Request, res: Response) => {
  try {
    const { globalStats, userStats, revenue } = req.body;
    const userId = req.headers["user-id"] as string;

    if (!userId) {
      return res.status(401).json({ message: "Unauthorized" });
    }

    // Validation
    if (revenue && (typeof revenue !== 'number' || revenue < 0)) {
      return res.status(400).json({ message: "Invalid revenue value" });
    }

    const updateData = {
      globalStats: {
        totalUsers: globalStats?.totalUsers || 0,
        activeUsers: globalStats?.activeUsers || 0,
        storageUsed: globalStats?.storageUsed || 0,
        monthlyVisitors: globalStats?.monthlyVisitors || 0
      },
      userStats: {
        averageSession: userStats?.averageSession || 0,
        retentionRate: userStats?.retentionRate || 0,
        popularFeatures: userStats?.popularFeatures || []
      },
      revenue: revenue || 0,
      updatedBy: userId
    };

    const updatedStats = await Statistics.findOneAndUpdate(
      {},
      updateData,
      { 
        new: true,
        upsert: true,
        setDefaultsOnInsert: true
      }
    ).populate('updatedBy', 'username email role');

    // Async logging and notification without blocking response
    try {
      await Log.create({
        action: "UPDATE_STATS",
        performedBy: userId,
        details: {
          revenue: updatedStats.revenue,
          globalStats: updatedStats.globalStats,
          userStats: updatedStats.userStats
        },
      });

      const admins = await User.find({ role: "SuperAdmin" });
      await Promise.all(admins.map(admin => 
        Notification.create({
          userId: admin._id,
          title: "Statistics Updated",
          message: `Platform statistics updated`,
          type: "info",
        })
      ));
    } catch (logError) {
      console.error("Logging failed:", logError);
    }

    return res.status(200).json({
      message: "Statistics updated successfully",
      statistics: updatedStats
    });

  } catch (error) {
    console.error("Error updating statistics:", error);
    return res.status(500).json({ message: "Failed to update statistics" });
  }
};