import { Request, Response } from "express";
import { SalesZone } from "../models/SalesZone";
import { Log } from "../models/Log";
import { Notification } from "../models/Notification";
import { User } from "../models/User";
import mongoose from "mongoose";

export const createZone = async (req: Request, res: Response) => {
  try {
    const { name, city, country, address, basePrice } = req.body;
    const userId = req.headers["user-id"] as string;

    // Validate headers
    if (!userId || !mongoose.Types.ObjectId.isValid(userId)) {
      return res.status(401).json({ message: "Invalid authentication credentials" });
    }

    // Validate required fields
    const requiredFields = ['name', 'city', 'country', 'address', 'basePrice'];
    const missingFields = requiredFields.filter(field => !req.body[field]);
    
    if (missingFields.length > 0) {
      return res.status(400).json({
        message: `Missing required fields: ${missingFields.join(', ')}`,
        fields: missingFields
      });
    }

    // Validate basePrice type
    if (typeof basePrice !== 'number' || basePrice <= 0) {
      return res.status(400).json({ 
        message: "Base price must be a positive number",
        field: 'basePrice'
      });
    }

    // Create new zone
    const newZone = new SalesZone({
      name: name.trim(),
      city: city.trim(),
      country: country.trim(),
      address: address.trim(),
      basePrice,
      photographers: [],
      images: []
    });

    // Validate Mongoose schema
    const validationError = newZone.validateSync();
    if (validationError) {
      return res.status(400).json({
        message: "Validation failed",
        errors: validationError.errors
      });
    }

    await newZone.save();

    return res.status(201).json({
      message: "Sales zone created successfully",
      zone: {
        id: newZone._id,
        name: newZone.name,
        city: newZone.city,
        country: newZone.country,
        address: newZone.address,
        basePrice: newZone.basePrice
      }
    });

  } catch (error: unknown) {
    console.error("Error creating zone:", error);
    
    // Add proper type checking
    if (error instanceof mongoose.Error.ValidationError) {
      return res.status(400).json({
        message: "Validation error",
        errors: error.errors
      });
    }
    
    // Handle other errors safely
    return res.status(500).json({
      message: "Internal server error",
      error: process.env.NODE_ENV === 'development' && error instanceof Error 
        ? error.message 
        : undefined
    });
  }
};

export const updateZone = async (req: Request, res: Response) => {
  const { id } = req.params;
  const userId = req.headers["user-id"] as string;

  if (!userId || !mongoose.Types.ObjectId.isValid(userId)) {
    return res.status(400).json({ message: "Invalid User ID" });
  }

  const zone = await SalesZone.findById(id);
  if (!zone) {
    return res.status(404).json({ message: "Zone not found" });
  }

  const updatedZone = await SalesZone.findByIdAndUpdate(id, req.body, { new: true });

  return res.status(200).json({ message: "Zone updated successfully", zone: updatedZone });
};

// controllers/zone.ts
export const updateZoneImages = async (req: Request, res: Response) => {
  const { id } = req.params;
  const { images } = req.body;

  try {
    const zone = await SalesZone.findByIdAndUpdate(
      id,
      { $push: { images: { $each: images } } },
      { new: true }
    );
    
    res.json(zone);
  } catch (error) {
    res.status(500).json({ message: 'Update failed' });
  }
};

export const deleteZone = async (req: Request, res: Response) => {
  const { id } = req.params;
  const userId = req.headers["user-id"] as string;

  if (!userId || !mongoose.Types.ObjectId.isValid(userId)) {
    return res.status(400).json({ message: "Invalid User ID" });
  }

  try {
    const deletedZone = await SalesZone.findByIdAndDelete(id);
    if (!deletedZone) {
      return res.status(404).json({ message: "Sales Zone not found" });
    }

    await Log.create({
      action: "DELETE_ZONE",
      performedBy: new mongoose.Types.ObjectId(userId),
      details: { zoneId: id, name: deletedZone.name },
    });

    return res.status(200).json({ message: "Sales Zone deleted successfully" });
  } catch (error) {
    console.error("Error deleting sales zone:", error);
    return res.status(500).json({ message: "Failed to delete sales zone" });
  }
};

export const getAllZones = async (req: Request, res: Response) => {
  try {
    const userId = req.headers["user-id"] as string;
    if (!userId || !mongoose.Types.ObjectId.isValid(userId)) {
      return res.status(400).json({ message: "Invalid User ID" });
    }

    const user = await User.findById(new mongoose.Types.ObjectId(userId));
    if (!user) return res.status(404).json({ message: "User not found" });

    const zones = await SalesZone.find()
      .populate("photographers", "name email")
      .populate("images") // Added image population
      .sort({ createdAt: -1 });

    return res.status(200).json(zones);
  } catch (error) {
    console.error("Error fetching sales zones:", error);
    return res.status(500).json({ message: "Failed to fetch sales zones" });
  }
};

export const getZoneById = async (req: Request, res: Response) => {
  try {
    const zone = await SalesZone.findById(req.params.id)
      .populate("photographers", "name email")
      .populate("images"); // Added image population

    if (!zone) {
      return res.status(404).json({ message: "Zone not found" });
    }

    res.json(zone);
  } catch (err) {
    res.status(500).json({ error: (err as Error).message });
  }
};
