import { Request, Response } from 'express';
import { Product } from '../models/Product';
import { Log } from '../models/Log';
import { Notification } from '../models/Notification';
import { User } from 'src/models/User';

// Create a new product
export const createProduct = async (req: Request, res: Response) => {
  try {
    const product = new Product(req.body);
    const savedProduct = await product.save();

    res.status(201).json(savedProduct);
  } catch (err) {
    res.status(400).json({ error: (err as Error).message });
  }
};

// Get a product by ID
export const getProductById = async (req: Request, res: Response) => {
  try {
    const product = await Product.findById(req.params.id);
    if (!product) {
      return res.status(404).json({ message: 'Product not found' });
    }
    res.json(product);
  } catch (err) {
    res.status(500).json({ error: (err as Error).message });
  }
};

// Get all products
export const getAllProducts = async (req: Request, res: Response) => {
  try {
     // Extract userId and userRole from request headers
     const userId = req.headers["user-id"] as string; // Access userId from headers
     const userRole = req.headers["user-role"] as string; // Access userRole from headers
 
     if (!userId) {
         return res.status(401).json({ message: "Unauthorized: No user ID provided" });
     }
 
     // If the user is SuperAdmin, show all notifications
     if (userRole === "SuperAdmin" || userRole === "Photographer") {
      const products = await Product.find().sort({ createdAt: -1 });
         return res.status(200).json(products);
     }
     
    const products = await Product.find({ userId }).sort({ createdAt: -1 });
    res.json(products);
  } catch (err) {
    res.status(500).json({ error: (err as Error).message });
  }
};

// Update a product by ID
export const updateProduct = async (req: Request, res: Response) => {
  try {
    const updatedProduct = await Product.findByIdAndUpdate(req.params.id, req.body, { new: true });
    if (!updatedProduct) {
      return res.status(404).json({ message: 'Product not found' });
    }

    // Log update
    await Log.create({
      action: 'UPDATE',
      performedBy: req.user?._id,
      details: { productId: updatedProduct._id, updatedFields: req.body },
    });

    // Send notification
    await Notification.create({
      userId: req.user?._id, // Change as needed for recipient
      title: 'Product Updated',
      message: `The product (${updatedProduct.title}) has been updated.`,
      type: 'info',
    });

    res.json(updatedProduct);
  } catch (err) {
    res.status(500).json({ error: (err as Error).message });
  }
};

// Delete a product by ID
export const deleteProduct = async (req: Request, res: Response) => {
  try {
    const deletedProduct = await Product.findByIdAndDelete(req.params.id);
    if (!deletedProduct) {
      return res.status(404).json({ message: 'Product not found' });
    }

    // Log deletion
    await Log.create({
      action: 'DELETE',
      performedBy: req.user?._id,
      details: { productId: deletedProduct._id, name: deletedProduct.title },
    });

    // Send notification
    await Notification.create({
      userId: req.user?._id, // Change as needed for recipient
      title: 'Product Deleted',
      message: `The product (${deletedProduct.title}) has been deleted.`,
      type: 'error',
    });

    res.json({ message: 'Product deleted successfully' });
  } catch (err) {
    res.status(500).json({ error: (err as Error).message });
  }
};
