import { Request, Response } from "express";
import { Price } from "../models/Price";
import { Log } from "../models/Log";
import { Notification } from "../models/Notification";
import { User } from "../models/User";

// Create a new price entry
export const createPrice = async (req: Request, res: Response) => {
  try {
    const { amount, currency, photographerShare, platformShare, createdBy } = req.body;

    // Validate total share = 100%
    if (photographerShare + platformShare !== 100) {
      return res.status(400).json({ message: "Total revenue share must be 100%" });
    }

    const newPrice = new Price({
      amount,
      currency,
      photographerShare,
      platformShare,
      createdBy,
    });

    await newPrice.save();

    // Log the creation
    await Log.create({
      action: "CREATE_PRICE",
      performedBy: createdBy,
      details: { amount, currency, photographerShare, platformShare },
    });

    // Notify admins about new price creation
    const admins = await User.find({ role: "SuperAdmin" });
    for (const admin of admins) {
      await Notification.create({
        userId: admin._id,
        title: "New Price Created",
        message: `A new price of ${amount} ${currency} has been set.`,
        type: "info",
      });
    }

    return res.status(201).json({ message: "Price created successfully", price: newPrice });
  } catch (error) {
    console.error("Error creating price:", error);
    return res.status(500).json({ message: "Failed to create price" });
  }
};

// Update an existing price entry
export const updatePrice = async (req: Request, res: Response) => {
  const { id } = req.params;
  const { amount, currency, photographerShare, platformShare, updatedBy } = req.body;

  try {
    const updatedPrice = await Price.findByIdAndUpdate(
      id,
      { amount, currency, photographerShare, platformShare },
      { new: true }
    );

    if (!updatedPrice) {
      return res.status(404).json({ message: "Price not found" });
    }

    // Log the update action
    await Log.create({
      action: "UPDATE_PRICE",
      performedBy: updatedBy,
      details: { priceId: id, updatedFields: req.body },
    });

    // Notify the creator about the update
    await Notification.create({
      userId: updatedPrice.createdBy,
      title: "Price Updated",
      message: `Your price of ${amount} ${currency} has been updated.`,
      type: "info",
    });

    return res.status(200).json({ message: "Price updated successfully", price: updatedPrice });
  } catch (error) {
    console.error("Error updating price:", error);
    return res.status(500).json({ message: "Failed to update price" });
  }
};

// Delete a price entry
export const deletePrice = async (req: Request, res: Response) => {
  const { id } = req.params;

  try {
    const deletedPrice = await Price.findByIdAndDelete(id);

    if (!deletedPrice) {
      return res.status(404).json({ message: "Price not found" });
    }

    // Log the deletion action
    await Log.create({
      action: "DELETE_PRICE",
      performedBy: req.user?._id, // Assuming req.user is set
      details: { priceId: id, amount: deletedPrice.amount, currency: deletedPrice.currency },
    });

    // Notify the creator about the deletion
    await Notification.create({
      userId: deletedPrice.createdBy,
      title: "Price Deleted",
      message: `Your price of ${deletedPrice.amount} ${deletedPrice.currency} has been removed.`,
      type: "error",
    });

    return res.status(200).json({ message: "Price deleted successfully" });
  } catch (error) {
    console.error("Error deleting price:", error);
    return res.status(500).json({ message: "Failed to delete price" });
  }
};

// Get all price entries
export const getAllPrices = async (_req: Request, res: Response) => {
  try {
    const prices = await Price.find().populate("createdBy", "username email");
    return res.status(200).json(prices);
  } catch (error) {
    console.error("Error fetching prices:", error);
    return res.status(500).json({ message: "Failed to fetch prices" });
  }
};

// Get a single price entry by ID
export const getPriceById = async (req: Request, res: Response) => {
  try {
    const price = await Price.findById(req.params.id).populate("createdBy", "username email");
    if (!price) {
      return res.status(404).json({ message: "Price not found" });
    }
    return res.status(200).json(price);
  } catch (error) {
    console.error("Error fetching price:", error);
    return res.status(500).json({ message: "Failed to fetch price" });
  }
};
