import { Request, Response } from "express";
import { Notification } from "../models/Notification";
import { RequestHandler } from "express";

// ✅ Controller: Fetch Notifications
export const getNotifications = async (req: Request, res: Response) => {
    try {
        // Extract userId and userRole from request headers
        const userId = req.headers["user-id"] as string; // Access userId from headers
        const userRole = req.headers["user-role"] as string; // Access userRole from headers

        if (!userId) {
            return res.status(401).json({ message: "Unauthorized: No user ID provided" });
        }

        // If the user is SuperAdmin, show all notifications
        if (userRole === "SuperAdmin") {
            const notifications = await Notification.find().sort({ createdAt: -1 });
            return res.status(200).json(notifications);
        }

        // Fetch notifications for the logged-in user
        const notifications = await Notification.find({ userId }).sort({ createdAt: -1 });
        return res.status(200).json(notifications);
    } catch (error) {
        console.error("Error fetching notifications:", error);
        return res.status(500).json({ message: "Internal server error" });
    }
};

// ✅ Controller: Mark Notifications as Read
export const markNotificationsAsRead = async (req: Request, res: Response) => {
    try {
        const userId = req.user?.id || req.headers["user-id"] as string; // Check for userId from `req.user` or headers
        const userRole = req.user?.role || req.headers["user-role"] as string; // Check for userRole from `req.user` or headers

        if (!userId) {
            return res.status(401).json({ message: "Unauthorized: No user ID provided" });
        }

        // If the user is a SuperAdmin, mark all notifications as read
        if (userRole === "SuperAdmin") {
            await Notification.updateMany({ read: false }, { $set: { read: true } });
            return res.status(200).json({ message: "All notifications for all users marked as read" });
        }

        // Otherwise, mark notifications only for the current user
        await Notification.updateMany({ userId, read: false }, { $set: { read: true } });
        return res.status(200).json({ message: "All your notifications marked as read" });
    } catch (error) {
        console.error("Error marking notifications as read:", error);
        return res.status(500).json({ message: "Internal server error" });
    }
};

// Create a new notification (can be used for testing or specific events)
export const createNotification = async (req: Request, res: Response) => {
    try {
        const { userId, title, message, type } = req.body;

        if (!userId || !title || !message || !type) {
            return res.status(400).json({ message: "All fields are required" });
        }

        const newNotification = new Notification({
            userId,
            title,
            message,
            type,
        });

        await newNotification.save();
        return res.status(201).json(newNotification);
    } catch (error) {
        console.error("Error creating notification:", error);
        return res.status(500).json({ message: "Internal server error" });
    }
};
