import { Request, Response } from "express";
import { Log } from "../models/Log";
import { User } from "../models/User"; // Assuming you have a User model
import mongoose from "mongoose";

// Get all logs with optional filters and pagination
export const getLogs = async (req: Request, res: Response) => {
  try {
    const { action, performedBy, page = 1, limit = 1000000 } = req.query;
    const filter: any = {};

    // Read User-Id from headers
    const userId = req.headers["user-id"] as string;
    const roleUser = req.headers["role-user"] as string;

    // Validate User-Id
    if (!userId || !mongoose.Types.ObjectId.isValid(userId)) {
      return res.status(400).json({ message: "Invalid or missing User-Id" });
    }

    // Get user from database
    const user = await User.findById(userId);
    if (!user) {
      return res.status(404).json({ message: "User not found" });
    }

    // Apply filters
    if (action) filter.action = action;
    if (performedBy) filter.performedBy = performedBy;

    // Fetch logs with pagination
    const logs = await Log.find(filter)
      .populate("performedBy", "username email")
      .sort({ timestamp: -1 })
      .skip((Number(page) - 1) * Number(limit))
      .limit(Number(limit));

    const totalLogs = await Log.countDocuments(filter);

    return res.status(200).json({
      logs,
      total: totalLogs,
      page: Number(page),
      limit: Number(limit),
    });
  } catch (error) {
    console.error("Error fetching logs:", error);
    return res.status(500).json({ message: "Failed to fetch logs" });
  }
};

// Create a new log entry
export const createLog = async (req: Request, res: Response) => {
  const { action, performedBy, details } = req.body;

  // Validate request body
  if (!action || !performedBy) {
    return res.status(400).json({ message: "Action and performedBy are required" });
  }

  try {
    const newLog = new Log({ action, performedBy, details });
    await newLog.save();
    return res.status(201).json({ message: "Log created successfully", log: newLog });
  } catch (error) {
    console.error("Error creating log:", error);
    return res.status(500).json({ message: "Failed to create log" });
  }
};