import { Request, Response } from 'express';
import mongoose from 'mongoose';
import { Image } from '../models/Image';
import sharp from 'sharp';
import { v4 as uuidv4 } from 'uuid';
import path from 'path';
import fs from 'fs';
import { Log } from '../models/Log';
import { Notification } from '../models/Notification';

const UPLOAD_DIR = path.join(__dirname, '../public/uploads');

export const uploadImages = async (req: Request, res: Response) => {
  try {
    // Validate authentication and files
    const userId = req.headers["user-id"] as string;

    if (!userId) return res.status(403).json({ message: 'Unauthorized' });
    if (!req.files || !Array.isArray(req.files)) {
      return res.status(400).json({ message: 'No files uploaded' });
    }

    console.log("req.files", req.files);
    
    const files = req.files as Express.Multer.File[];
    const { zone: salesZoneId } = req.body;

    // Validate salesZoneId
    if (!salesZoneId || !mongoose.Types.ObjectId.isValid(salesZoneId)) {
      return res.status(400).json({ message: 'Valid SalesZone ID is required' });
    }

    const uploadedImages = [];
    const zoneUploadDir = path.join(UPLOAD_DIR, salesZoneId);
    const zoneThumbnailDir = path.join(zoneUploadDir, 'thumbnails');

    // Create directories if needed
    [zoneUploadDir, zoneThumbnailDir].forEach(dir => {
      if (!fs.existsSync(dir)) fs.mkdirSync(dir, { recursive: true });
    });

    // Process each file
    for (const file of files) {
      const imageId = new mongoose.Types.ObjectId();
      const originalName = file.originalname
        .replace(path.extname(file.originalname), '')
        .replace(/[^a-zA-Z0-9]/g, '-')
        .replace(/-+/g, '-')
        .toLowerCase();

      const fileExt = path.extname(file.originalname);
      const originalFilename = `${imageId}-${originalName}${fileExt}`;
      const thumbnailFilename = `${imageId}-thumb-${originalName}${fileExt}`;

      const originalPath = path.join(zoneUploadDir, originalFilename);
      const thumbnailPath = path.join(zoneThumbnailDir, thumbnailFilename);

      try {
        // Process image with Sharp
        const imageProcessor = sharp(file.buffer);
        const metadata = await imageProcessor.metadata();

        // Save files
        await Promise.all([
          imageProcessor.toFile(originalPath),
          sharp(file.buffer)
            .resize(800, 800, { 
              fit: 'inside', 
              withoutEnlargement: true 
            })
            .toFile(thumbnailPath)
        ]);

        // Create image document
        const newImage = new Image({
          _id: imageId,
          originalUrl: `/uploads/${salesZoneId}/${originalFilename}`,
          thumbnailUrl: `/uploads/${salesZoneId}/thumbnails/${thumbnailFilename}`,
          uploadedBy: userId,
          metadata: {
            size: file.size,
            format: metadata.format || 'unknown',
            resolution: metadata.width && metadata.height ? 
              `${metadata.width}x${metadata.height}` : 'N/A'
          },
          tags: req.body.tags ? req.body.tags.split(',') : [],
          zone: salesZoneId,
          file: req.body.file || null
        });

        await newImage.save();
        uploadedImages.push(newImage);

        // Create log and notification
        await Promise.all([
          Log.create({
            action: 'UPLOAD',
            performedBy: userId,
            details: { imageId, originalUrl: newImage.originalUrl }
          }),
          Notification.create({
            userId: userId,
            title: 'Image Uploaded',
            message: `Image ${originalName} uploaded to zone ${salesZoneId}`,
            type: 'success'
          })
        ]);

      } catch (error) {
        // Cleanup failed files
        [originalPath, thumbnailPath].forEach(path => {
          if (fs.existsSync(path)) fs.unlinkSync(path);
        });
        throw error;
      }
    }

    res.status(201).json(uploadedImages);
  } catch (err) {
    res.status(500).json({
      error: (err as Error).message,
      message: 'Image upload failed'
    });
  }
};

// ✅ Get an image by ID
export const getImageById = async (req: Request, res: Response) => {
  try {
    const image = await Image.findById(req.params.id).populate('uploadedBy', 'name email');
    if (!image) return res.status(404).json({ message: 'Image not found' });

    res.json(image);
  } catch (err: unknown) {
    const error = err as Error;
    res.status(500).json({ error: error.message });
  }
};

// ✅ Get all images
export const getAllImages = async (_req: Request, res: Response) => {
  try {
    const images = await Image.find().populate('uploadedBy', 'name email');
    res.json(images);
  } catch (err) {
    res.status(500).json({ error: (err as Error).message });
  }
};

// ✅ Update image metadata
export const updateImage = async (req: Request, res: Response) => {
  try {
    const updatedImage = await Image.findByIdAndUpdate(req.params.id, req.body, { new: true });
    if (!updatedImage) return res.status(404).json({ message: 'Image not found' });

    // ✅ Ensure `req.user` exists before accessing `_id`
    if (!req.user) return res.status(403).json({ message: 'Unauthorized' });

    await Log.create({
      action: 'UPDATE',
      performedBy: req.user._id,
      details: { imageId: updatedImage._id, updatedFields: req.body },
    });

    res.json(updatedImage);
  } catch (err) {
    res.status(500).json({ error: (err as Error).message });
  }
};

// ✅ Delete an image
export const deleteImage = async (req: Request, res: Response) => {
  try {
    const deletedImage = await Image.findByIdAndDelete(req.params.id);
    if (!deletedImage) return res.status(404).json({ message: 'Image not found' });

    // ✅ Ensure `req.user` exists before accessing `_id`
    if (!req.user) return res.status(403).json({ message: 'Unauthorized' });

    await Log.create({
      action: 'DELETE',
      performedBy: req.user._id,
      details: { imageId: deletedImage._id, originalUrl: deletedImage.originalUrl },
    });

    // ✅ Send a notification
    await Notification.create({
      userId: req.user._id,
      title: 'Image Deleted',
      message: `The image (${deletedImage.originalUrl}) has been deleted.`,
      type: 'error',
    });

    res.json({ message: 'Image deleted successfully' });
  } catch (err) {
    res.status(500).json({ error: (err as Error).message });
  }
};
