import { Request, Response } from "express";
import { Log } from "../models/Log";
import { Notification } from "../models/Notification";
// src/controllers/fileController.ts
import { File } from '../models/File';
import mongoose from 'mongoose';
// Corrected imports in fileController.ts
import { Status } from '../models/Status'; // Remove IStatus
import { SalesZone, ISalesZone } from "../models/SalesZone";
import { User, IUser } from "../models/User";
import sharp from 'sharp';
import { Image } from '../models/Image'; // Ensure Image model is imported


// controllers/file.ts
export const createFile = async (req: Request, res: Response) => {
  const {
    clientName,
    email,
    phone,
    address,
    projectType,
    product,
    salesZone,
    delivery,
    total,
    currency,
    assignedTo,
    status, // Status reference
  } = req.body;

  try {
    // Validate required fields
    if (!clientName || !projectType || !status) {
      return res.status(400).json({ message: 'Missing required fields' });
    }

    // Validate status exists
    const statusExists = await Status.findById(status);
    if (!statusExists) {
      return res.status(400).json({ message: 'Invalid status reference' });
    }

    // Create the file
    const newFile = new File({
      clientName,
      email,
      phone,
      address,
      projectType,
      product,
      salesZone,
      delivery,
      total,
      currency,
      assignedTo,
      status,
    });

    await newFile.save();


    // Read User-Id from headers
    const userId = req.headers["user-id"] as string;

    // Validate User-Id
    if (!userId || !mongoose.Types.ObjectId.isValid(userId)) {
      return res.status(400).json({ message: "Invalid or missing User-Id in headers" });
    }

    // Log file creation
    await Log.create({
      action: "CREATE_FILE",
      performedBy: userId,
      details: { fileId: newFile._id, clientName: newFile.clientName },
    });

    // Notify assigned user
    if (assignedTo) {
      await Notification.create({
        userId: assignedTo,
        title: "New File Created",
        message: `A new file for project: ${projectType} has been created.`,
        type: "info",
      });
    }

    return res.status(201).json(newFile);
  } catch (error) {
    console.error('Error creating file:', error);

    // Handle specific Mongoose validation errors
    if (error instanceof Error && error.name === 'ValidationError') {
      return res.status(400).json({ message: error.message });
    }

    // Handle other errors
    if (error instanceof Error) {
      return res.status(500).json({ message: error.message });
    }

    // Fallback for unknown errors
    return res.status(500).json({ message: 'Failed to create file' });
  }
};

// Add to your existing file controller
export const getFileStatistics = async (req: Request, res: Response) => {
  try {
    const userId = req.headers["user-id"] as string;
    const userRole = req.headers["user-role"] as string;

    // Reuse the same query logic from getFiles
    let query: any = {};

    if (userRole === "Photographer") {
      const salesZones = await SalesZone.find({ photographers: userId }).select("_id").lean();
      const salesZoneIds = salesZones.map(zone => zone._id);
      if (salesZoneIds.length === 0) return res.status(200).json(getEmptyStats());
      query.salesZone = { $in: salesZoneIds };
    }

    // Add Imprimante role filter
    if (userRole === "Imprimante") {
      const paidStatus = await Status.findOne({ status: "Paid" }).select("_id").lean();
      if (paidStatus) {
        query.status = paidStatus._id;
      }
    }

    const stats = await File.aggregate([
      { $match: query },
      { 
        $lookup: {
          from: "statuses",
          localField: "status",
          foreignField: "_id",
          as: "statusData"
        }
      },
      { $unwind: "$statusData" },
      {
        $group: {
          _id: null,
          total: { $sum: 1 },
          paid: { 
            $sum: { 
              $cond: [{ $eq: ["$statusData.status", "Paid"] }, 1, 0] 
            } 
          },
          pending: { 
            $sum: { 
              $cond: [{ $eq: ["$statusData.status", "Pending"] }, 1, 0] 
            } 
          },
          inProgress: { 
            $sum: { 
              $cond: [{ $eq: ["$statusData.status", "In Progress"] }, 1, 0] 
            } 
          },
          completed: { 
            $sum: { 
              $cond: [{ $eq: ["$statusData.status", "Completed"] }, 1, 0] 
            } 
          }
        }
      },
      {
        $project: {
          _id: 0,
          total: 1,
          paid: 1,
          pending: 1,
          inProgress: 1,
          completed: 1
        }
      }
    ]);

    // Log the statistics request
    await Log.create({
      action: "FETCH_STATS",
      performedBy: userId,
      details: { userRole, stats: stats[0] || {} }
    });

    return res.status(200).json(stats[0] || getEmptyStats());
  } catch (error) {
    console.error("Error fetching statistics:", error);
    return res.status(500).json({ message: "Failed to fetch statistics" });
  }
};

// Helper function for empty stats
const getEmptyStats = () => ({
  total: 0,
  paid: 0,
  pending: 0,
  inProgress: 0,
  completed: 0
});

export const updateFile = async (req: Request, res: Response) => {
  const { id } = req.params;

  try {
    // Accept full update operations
    const updatedFile = await File.findByIdAndUpdate(
      id,
      req.body, // Directly use the request body
      { new: true, runValidators: true }
    ).populate('images salesZone product');

    if (!updatedFile) {
      return res.status(404).json({ message: 'File not found' });
    }

    res.json(updatedFile);
  } catch (error) {
    res.status(500).json({ message: 'Update failed', error });
  }
};
// Helper function should be at the bottom if using function hoisting
const getImageResolution = async (path: string) => {
  const metadata = await sharp(path).metadata();
  return `${metadata.width}x${metadata.height}`;
};

// Delete a file
export const deleteFile = async (req: Request, res: Response) => {
  const { id } = req.params;

  try {
    const deletedFile = await File.findByIdAndDelete(id);

    if (!deletedFile) {
      return res.status(404).json({ message: "File not found" });
    }

    // Log deletion
    await Log.create({
      action: "DELETE_FILE",
      performedBy: req.user?._id,
      details: { fileId: id, clientName: deletedFile.clientName },
    });

    // Notify assigned user
    if (deletedFile.assignedTo) {
      await Notification.create({
        userId: deletedFile.assignedTo,
        title: "File Deleted",
        message: `The file for project: ${deletedFile.projectType} has been deleted.`,
        type: "error",
      });
    }

    return res.status(200).json({ message: "File deleted successfully", file: deletedFile });
  } catch (error) {
    console.error("Error deleting file:", error);
    return res.status(500).json({ message: "Failed to delete file" });
  }
};

export const getFiles = async (req: Request, res: Response) => {
  try {
    const userId = req.headers["user-id"] as string;
    const userRole = req.headers["user-role"] as string;

    if (!userId || !mongoose.Types.ObjectId.isValid(userId)) {
      return res.status(400).json({ message: "Invalid User-Id" });
    }

    let query: any = {};

    if (userRole === "Photographer") {
      const salesZones = await SalesZone.find({ photographers: userId }).select("_id").lean();
      const salesZoneIds = salesZones.map(zone => zone._id);

      if (salesZoneIds.length === 0) return res.status(200).json({ files: [] });

      query.salesZone = { $in: salesZoneIds };
    }

    const files = await File.find(query)
      .populate<{ status: typeof Status }>("status", "name color order")
      .populate<{ salesZone: ISalesZone }>("salesZone", "name city")
      .populate<{ assignedTo: IUser }>("assignedTo", "firstName lastName email")
      .lean();

    await Log.create({
      action: "FETCH_FILES",
      performedBy: userId,
      details: { count: files.length, userRole },
    });

    return res.status(200).json({ files });
  } catch (error) {
    console.error("Error fetching files:", error);
    return res.status(500).json({ message: "Failed to fetch files" });
  }
};

export const getFileById = async (req: Request, res: Response) => {
  const { id } = req.params;

  try {
    // Validate file ID format
    if (!mongoose.Types.ObjectId.isValid(id)) {
      return res.status(400).json({ message: "Invalid file ID format" });
    }

    // Get user credentials from headers
    const userId = req.headers["user-id"] as string;
    const userRole = req.headers["user-role"] as string;

    // Validate user credentials
    if (!userId || !mongoose.Types.ObjectId.isValid(userId)) {
      return res.status(401).json({ message: "Invalid or missing User-Id" });
    }

    if (!["SuperAdmin", "Photographer"].includes(userRole)) {
      return res.status(401).json({ message: "Unauthorized role" });
    }

    // In fileController.ts
    const file = await File.findById(id)
      .populate('images') // Add this line
      .populate('status', 'name color order')
      .populate<{ salesZone: ISalesZone }>('salesZone', 'name city')
      .populate<{ assignedTo: IUser }>('assignedTo', 'firstName lastName email')
      .lean();

    if (!file) {
      return res.status(404).json({ message: "File not found" });
    }

    // Authorization check for Photographers
    if (userRole === "Photographer") {
      const allowedZones = await SalesZone.find({
        photographers: userId
      }).select('_id').lean();

      const allowedZoneIds = allowedZones.map(zone => zone._id.toString());

      if (!allowedZoneIds.includes(file.salesZone?._id.toString())) {
        return res.status(403).json({ message: "Access denied to this file" });
      }
    }

    // Log the access
    await Log.create({
      action: "VIEW_FILE",
      performedBy: userId,
      details: { fileId: id }
    });

    return res.status(200).json(file);

  } catch (error) {
    console.error("Error fetching file:", error);

    // Handle specific errors
    if (error instanceof mongoose.Error.CastError) {
      return res.status(400).json({ message: "Invalid file ID format" });
    }

    const errorMessage = error instanceof Error ? error.message : "Failed to fetch file";
    return res.status(500).json({ message: errorMessage });
  }
};