import { Request, Response } from 'express';
import { Event } from '../models/Event';
import { Log } from '../models/Log';
import { Notification } from '../models/Notification';

import { isURL } from 'validator'; // If using the 'validator' package
import { User } from 'src/models/User';


// Create a new event
export const createEvent = async (req: Request, res: Response) => {
  try {
    const { name, location, date, photographers, basePrice, image } = req.body;

    // Check if the image is provided and valid, otherwise skip the validation
    if (image && !isURL(image)) {
      return res.status(400).json({ message: "Invalid image URL" });
    }

    // Extract userId from request headers
    const userId = req.headers["user-id"] as string;

    if (!userId) {
      return res.status(400).json({ message: "User ID is required" });
    }

    const newEvent = new Event({
      name,
      location,
      date,
      photographers: photographers || [],
      basePrice,
      image: image || undefined,  // If image is not provided, it will be undefined
    });



    const savedEvent = await newEvent.save();

    // Log creation
    await Log.create({
      action: 'CREATE', 
      performedBy: userId,
      details: { name: savedEvent.name, date: savedEvent.date, location: savedEvent.location },
    });

    // Notify admins about the new zone
    const admins = await User.find({ role: "SuperAdmin" });
    for (const admin of admins) {
      await Notification.create({
        userId: admin._id,
        title: "New Event Created",
        message: `An event (${savedEvent.name}) has been created for ${savedEvent.date}.`,
        type: "info",
      });
    }

    res.status(201).json(savedEvent);
  } catch (err) {
    res.status(400).json({ error: (err as Error).message });
  }
};

// Get an event by ID
export const getEventById = async (req: Request, res: Response) => {
  try {
    const event = await Event.findById(req.params.id);
    if (!event) {
      return res.status(404).json({ message: 'Event not found' });
    }
    res.json(event);
  } catch (err) {
    res.status(500).json({ error: (err as Error).message });
  }
};

// Get all events
export const getAllEvents = async (_req: Request, res: Response) => {
  try {
    const events = await Event.find();
    res.json(events);
  } catch (err) {
    res.status(500).json({ error: (err as Error).message });
  }
};

// Update an event by ID
export const updateEvent = async (req: Request, res: Response) => {
  const { id } = req.params;
  const { name, location, date, photographers, basePrice, image } = req.body;

  const userId = req.headers["user-id"] as string;
  
  if (!userId) {
    return res.status(400).json({ message: "User ID is required" });
  }

  try {
    const updatedEvent = await Event.findByIdAndUpdate(
      { name, location, date, photographers, basePrice, image },
      { new: true }
    );

    if (!updatedEvent) {
      return res.status(404).json({ message: 'Event not found' });
    }

    // Log update
    await Log.create({
      action: 'UPDATE',
      performedBy: userId,
      details: { eventId: updatedEvent._id, updatedFields: req.body },
    });

    // Notify admins about the new zone
    const admins = await User.find({ role: "SuperAdmin" });
    for (const admin of admins) {
      await Notification.create({
        userId: admin._id,
        title: "Event Updated",
        message: `The event (${updatedEvent.name}) has been updated.`,
        type: "info",
      });
    }

    res.json(updatedEvent);
  } catch (err) {
    res.status(500).json({ error: (err as Error).message });
  }
};

// Delete an event by ID
export const deleteEvent = async (req: Request, res: Response) => {
  try {
    const deletedEvent = await Event.findByIdAndDelete(req.params.id);
    if (!deletedEvent) {
      return res.status(404).json({ message: 'Event not found' });
    }

    // Log deletion
    await Log.create({
      action: 'DELETE',
      performedBy: req.user?._id,
      details: { eventId: deletedEvent._id, name: deletedEvent.name },
    });

    // Send notification
    await Notification.create({
      userId: req.user?._id, // Adjust as needed
      title: 'Event Deleted',
      message: `The event (${deletedEvent.name}) has been deleted.`,
      type: 'error',
    });

    res.json({ message: 'Event deleted successfully' });
  } catch (err) {
    res.status(500).json({ error: (err as Error).message });
  }
};
